/****************************************************************************\
**
** Button.js
**
** Source code for handling buttons states.
**
\****************************************************************************/
var mouseCaptured = false; // true if captured by owner.
var insideOwner = false; // true if mouse is inside owner during click hold.

/**
 *
 * ButtonStates constructor
 *
 * This class provides facility to hide or show button state's child canvas.
 *
 * @param children --> Container which contains all the child containers
 *                     containing different states of button.
 *
 */
function ButtonStates(children)
{
	// Following members hold button state's container objects.
	this.mOnUp = null;
	this.mOnOver = null;
	this.mOnDown = null;

	// Following members set visibility of child containers of button.
	this.SetVisibility_OnUp = SetVisibility_OnUp;
	this.SetVisibility_OnOver = SetVisibility_OnOver;
	this.SetVisibility_OnDown = SetVisibility_OnDown;

	for (var index = 0; index < children.count; ++index) {
		var child = children.getItem(index);

		// Button state is identified by the value of the tag associated with child.
		if (child.tag == "OnUp") {
			this.mOnUp = child
		}
		else if (child.tag == "OnOver") {
			this.mOnOver = child;
		}
		else if (child.tag == "OnDown") {
			this.mOnDown = child;
		}
	}
}

/**
 *
 * ButtonStates::SetVisibility_OnUp
 *
 * Sets visibility of the container containing button's "OnUp" state elements.
 *
 * @param	show --> Send true to show "OnUp" state, false otherwise.
 *
 */
function SetVisibility_OnUp(show)
{
	if (this.mOnUp != null) {
		show ? this.mOnUp.Visibility = "Visible" : this.mOnUp.Visibility = "Collapsed";
	}
}


/**
 *
 * ButtonStates::SetVisibility_OnOver
 *
 * Sets visibility of the container containing button's "OnOver" state elements.
 *
 * @param	show --> Send true to show "OnOver" state, false otherwise.
 *
 */
function SetVisibility_OnOver(show)
{
	if (this.mOnOver != null) {
		show ? this.mOnOver.Visibility = "Visible" : this.mOnOver.Visibility = "Collapsed";
	}
}

/**
 *
 * ButtonStates::SetVisibility_OnDown
 *
 * Sets visibility of the canvas containing button's "OnDown" state elements.
 *
 * @param	show --> Send true to show "OnDown" state, false otherwise.
 *
 */
function SetVisibility_OnDown(show)
{
	if (this.mOnDown != null) {
		show ? this.mOnDown.Visibility = "Visible" : this.mOnDown.Visibility = "Collapsed";
	}
}

/**
 *
 * MouseEnterHandler
 *
 * Handles mouse enter event on button. This function hides "OnUp" state and shows
 * "OnOver" state.of the button if button has not captured the mouse. If button has
 * captured the mouse, "OnDown" state will be shown.
 *
 * @param	sender --> Event sender.
 * @param	args --> Event arguments.
 *
 */
function MouseEnterHandler(sender, args)
{
	var btnStates = new ButtonStates(sender.getParent().children);

	if (mouseCaptured) {
		insideOwner = true;
	}
	btnStates.SetVisibility_OnUp(false);
	btnStates.SetVisibility_OnOver(!mouseCaptured);
	btnStates.SetVisibility_OnDown(mouseCaptured);
}

/**
 *
 * MouseLeaveHandler
 *
 * Handles mouse leave event on button. This function shows "OnUp" state container
 * and hides all other state containers.
 *
 * @param	sender --> Event sender.
 * @param	args --> Event arguments.
 *
 */
function MouseLeaveHandler(sender,args)
{
	var btnStates = new ButtonStates(sender.getParent().children);

	insideOwner = false;
	btnStates.SetVisibility_OnUp(true);
	btnStates.SetVisibility_OnOver(false);
	btnStates.SetVisibility_OnDown(false);
}

/**
 *
 * MouseLeftButtonDownHandler
 *
 * Handles mouse down event on button. This function shows "OnDown" state container
 * and hides all other state containers.
 *
 * @param	sender --> Event sender.
 * @param	args --> Event arguments.
 *
 */
function MouseLeftButtonDownHandler(sender,args)
{
	var btnStates = new ButtonStates(sender.getParent().children);

	// Capture the mouse.
	sender.CaptureMouse();
	mouseCaptured = true;
	insideOwner = true;
	btnStates.SetVisibility_OnUp(false);
	btnStates.SetVisibility_OnOver(false);
	btnStates.SetVisibility_OnDown(true);
}

/**
 *
 * MouseLeftButtonUpHandler
 *
 * Handles mouse up event on button. This function shows "OnOver" state container
 * if mouse is still over the button. If mouse up event was recieved while mouse
 * was outside of button, "OnUp" state is shown.
 *
 * @param	sender --> Event sender.
 * @param	args --> Event arguments.
 *
 */
function MouseLeftButtonUpHandler(sender,args)
{
	var btnStates = new ButtonStates(sender.getParent().children);

	btnStates.SetVisibility_OnUp(!insideOwner);
	btnStates.SetVisibility_OnOver(insideOwner);
	btnStates.SetVisibility_OnDown(false);

	// Release mouse capture.
	sender.ReleaseMouseCapture();
	mouseCaptured = false;
	insideOwner = false;
}
